/* Shared library add-on to iptables to add MARK target support. */
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>

#include <iptables.h>
#include <linux/netfilter_ipv4/ip_tables.h>
/* For 64bit kernel / 32bit userspace */
#include "../include/linux/netfilter_ipv4/ipt_MARK_and_DSCP.h"

/* Function which prints out usage message. */
static void
help(void)
{
	printf(
"MARK target v%s options:\n"
"  --set-mark value                   Set nfmark value\n"
"  --and-mark value                   Binary AND the nfmark with value\n"
"  --or-mark  value                   Binary OR  the nfmark with value\n"
"  --set-dscp  value                  set dscp field with value\n"
"\n",
IPTABLES_VERSION);
}

static struct option opts[] = {
	{ "set-mark", 1, 0, '1' },
	{ "and-mark", 1, 0, '2' },
	{ "or-mark", 1, 0, '3' },
	{ "set-dscp", 1, 0, '4' },
	{ 0 }
};

int string_to_number_u8(const char *s, unsigned int min, unsigned int max,
                u_int8_t *ret)
{
        int result;
        unsigned long number;

        result = string_to_number_l(s, min, max, &number);
        *ret = (u_int8_t)number;

        return result;
}

/* Initialize the target. */
static void
init(struct ipt_entry_target *t, unsigned int *nfcache)
{
}
static void
final_check(unsigned int flags)
{
	if (!flags)
		exit_error(PARAMETER_PROBLEM,
		           "MARK target: Parameter --set/and/or-mark"
			   " is required");
}

/* Function which parses command options; returns true if it
   ate an option */
static int
parse_v1(int c, char **argv, int invert, unsigned int *flags,
	 const struct ipt_entry *entry,
	 struct ipt_entry_target **target)
{
        u_int8_t opt = 0;
	struct ipt_mark_target_info_v1 *markinfo
		= (struct ipt_mark_target_info_v1 *)(*target)->data;

	switch (c) {
	case '1':
	        markinfo->mode = IPT_MARK_SET;
                opt = IPT_MARK_SET;
		break;
	case '2':
	        markinfo->mode = IPT_MARK_AND;
                opt = IPT_MARK_AND;
		break;
	case '3':
	        markinfo->mode = IPT_MARK_OR;
                opt = IPT_MARK_OR;
		break;
	case '4':
	        markinfo->mode |= IPT_DSCP_SET;
                opt = IPT_DSCP_SET;
		break;

	default:
		return 0;
	}

#ifdef KERNEL_64_USERSPACE_32
	if ((c!='4') && string_to_number_ll(optarg, 0, 0,  &markinfo->mark))
#else
	if ((c!='4') &&string_to_number_l(optarg, 0, 0, &markinfo->mark))
#endif
		exit_error(PARAMETER_PROBLEM, "Bad MARK value `%s'", optarg);

#ifdef KERNEL_64_USERSPACE_32
	if ((c=='4') && string_to_number_u8(optarg, 0, 0,  &markinfo->dscp))
#else
	if ((c=='4') && string_to_number_u8(optarg, 0, 0, &markinfo->dscp))
#endif
		exit_error(PARAMETER_PROBLEM, "Bad DSCP value `%s'", optarg);

	if (*flags & opt)
		exit_error(PARAMETER_PROBLEM,
			   "MARK target: you use a target twice in one rule");
	*flags |= opt;
	return 1;
}

#ifdef KERNEL_64_USERSPACE_32
static void
print_mark(unsigned long long mark)
{
	printf("0x%llx ", mark);
}
#else
static void
print_mark(unsigned long mark)
{
	printf("0x%lx ", mark);
}
#endif

/* Prints out the targinfo. */
static void
print_v1(const struct ipt_ip *ip,
	 const struct ipt_entry_target *target,
	 int numeric)
{
	const struct ipt_mark_target_info_v1 *markinfo =
		(const struct ipt_mark_target_info_v1 *)target->data;

        printf(" MARK_and_DSCP");
        
	if (markinfo->mode & IPT_MARK_SET )
        {
            printf(" --set-mark ");
        }
        if (markinfo->mode & IPT_MARK_AND )
        {
            printf(" --and-mark ");
        }
        if (markinfo->mode & IPT_MARK_OR )
        {
            printf(" --or-mark ");
        }
        print_mark(markinfo->mark);

        if (markinfo->mode & IPT_DSCP_SET )
        {
            printf(" --set-dscp ");
        }
        print_mark(markinfo->dscp);
}

/* Saves the union ipt_targinfo in parsable form to stdout. */
static void
save_v1(const struct ipt_ip *ip, const struct ipt_entry_target *target)
{
	const struct ipt_mark_target_info_v1 *markinfo =
		(const struct ipt_mark_target_info_v1 *)target->data;

        printf(" MARK_and_DSCP");
        
	if (markinfo->mode & IPT_MARK_SET )
        {
            printf(" --set-mark ");
        }
        if (markinfo->mode & IPT_MARK_AND )
        {
            printf(" --and-mark ");
        }
        if (markinfo->mode & IPT_MARK_OR )
        {
            printf(" --or-mark ");
        }
        print_mark(markinfo->mark);

        if (markinfo->mode & IPT_DSCP_SET )
        {
            printf(" --set-dscp ");
        }
        print_mark(markinfo->dscp);
}


static
struct iptables_target mark_v1 = {
	.next		= NULL,
	.name		= "MARK_and_DSCP",
	.version	= IPTABLES_VERSION,
	.revision	= 0,
	.size		= IPT_ALIGN(sizeof(struct ipt_mark_target_info_v1)),
	.userspacesize	= IPT_ALIGN(sizeof(struct ipt_mark_target_info_v1)),
	.help		= &help,
	.init		= &init,
	.parse		= &parse_v1,
	.final_check	= &final_check,
	.print		= &print_v1,
	.save		= &save_v1,
	.extra_opts	= opts
};

void _init(void)
{
	register_target(&mark_v1);
}
